﻿namespace KeyServerIntegrationSample
{
    using System;
    using System.IO;
    using System.Net;
    using System.Net.Http;
    using System.Security.Cryptography;
    using System.Text;
    using Newtonsoft.Json;
    using Newtonsoft.Json.Linq;

    public class Program
    {
        public static void Main()
        {
            // To make Content Key Request we will need Key Server URL.
            const string keyServerUrl = "http://key-server-management.axtest.net/api/WidevineProtectionInfo";

            // Request to the Key Server must contain signer and signature.
            const string requestSigner = "widevine_test";

            // SigningKey and IV are used to create signature.
            var signingKey = new byte[]
            {
                0x1a, 0xe8, 0xcc, 0xd0, 0xe7, 0x98, 0x5c, 0xc0,
                0xb6, 0x20, 0x3a, 0x55, 0x85, 0x5a, 0x10, 0x34,
                0xaf, 0xc2, 0x52, 0x98, 0x0e, 0x97, 0x0c, 0xa9,
                0x0e, 0x52, 0x02, 0x68, 0x9f, 0x94, 0x7a, 0xb9
            };

            var iv = new byte[]
            {
                0xd5, 0x8c, 0xe9, 0x54, 0x20, 0x3b, 0x7c, 0x9a,
                0x9a, 0x9d, 0x46, 0x7f, 0x59, 0x83, 0x92, 0x49
            };

            // In order to make Content Key Request we need to make JSON data structure,
            // that contains the content ID and the tracks array, that contains track types.
            // There may be multiple tracks. Supported track type values: "AUDIO", "SD", "HD". 
            var contentKeyRequest = new
            {
                content_id = "MEIzNTBDMDgtNEJDQi00Qjk2LUE4NzMtOEMyNEY2RTk5MUM1",
                tracks = new[]
                {
                    new { type = "HD" }
                }
            };

            var contentKeyRequestJson = JsonConvert.SerializeObject(contentKeyRequest);

            Console.WriteLine("Content Key Request JSON:");
            Console.WriteLine(contentKeyRequestJson);

            // To send the Content Key Request to Key Server we need to wrap Content Key Request to 
            // JSON data structure that contains base64 encoded Content Key Request, signature and signer.
            // To produce signature we need to calculate hash of the Content Key Request using SHA-1 algorithm and 
            // then encrypt the hash bytes with Signing Key and IV using AES-CBC encryption algorithm.
            byte[] contentkeyRequestHash;

            using (var sha1 = new SHA1Managed())
            {
                contentkeyRequestHash = sha1.ComputeHash(Encoding.UTF8.GetBytes(contentKeyRequestJson));
            }

            Console.WriteLine("Content Key Request hash:");
            Console.WriteLine(BitConverter.ToString(contentkeyRequestHash).Replace("-", ""));

            byte[] encryptedSignature;

            using (var memoryStream = new MemoryStream())
            {
                using (var aesProvider = new AesCryptoServiceProvider())
                {
                    using (var cryptoStream = new CryptoStream(memoryStream,
                        aesProvider.CreateEncryptor(signingKey, iv), CryptoStreamMode.Write))
                    {
                        cryptoStream.Write(contentkeyRequestHash, 0, contentkeyRequestHash.Length);
                    }
                    aesProvider.Clear();
                }
                encryptedSignature = memoryStream.ToArray();
            }

            var encryptedSignatureBase64 = Convert.ToBase64String(encryptedSignature);

            var requestData = new
            {
                request = Convert.ToBase64String(Encoding.UTF8.GetBytes(contentKeyRequestJson)),
                signature = encryptedSignatureBase64,
                signer = requestSigner
            };

            Console.WriteLine("Request data:");
            Console.WriteLine(requestData);

            // To receive Content Keys we need to make HTTP POST request to the Key Server specifying 
            // created request data as the request body.
            using (var client = new HttpClient())
            {
                var response = client
                    .PostAsync(
                        keyServerUrl,
                        new StringContent(JsonConvert.SerializeObject(requestData), Encoding.UTF8, "application/json")
                    ).GetAwaiter().GetResult();

                if (response.StatusCode != HttpStatusCode.OK)
                {
                    Console.WriteLine($"Request failed. Status code: {response.StatusCode} Reason phrase: {response.ReasonPhrase}");
                    return;
                }

                var responseContent = response.Content.ReadAsStringAsync().GetAwaiter().GetResult();

                Console.WriteLine("Response:");
                Console.WriteLine(responseContent);

                try
                {
                    var contentKeyResponseJson = JObject.Parse(responseContent);
                    var responseAsBase64 = (string)contentKeyResponseJson["response"];

                    Console.WriteLine("Decoded response data:");
                    Console.WriteLine(Encoding.UTF8.GetString(Convert.FromBase64String(responseAsBase64)));
                }
                catch (Exception ex)
                {
                    Console.WriteLine($"Parsing response failed with exception message: {ex.Message}");
                }
            }
        }
    }
}